package cnrg.itx.signal;

import cnrg.itx.datax.*;
import cnrg.itx.ds.*;
import cnrg.itx.signal.SignalID;

public class SIPPacket extends SigPacket
{
	/** Source userID **/
	protected UserID source;
	/** Source location **/
	protected Location srcLoc;
	/** Source application's string description **/
	protected String appDesc = "";
	/** Result of query to user **/
	protected int resultFlag = SignalID.INVALID;
	/** Source's data exchange properties **/
	protected PropertiesCollection myPC = null;
	/** Reason for rejecting a call **/
	protected String reason = "";
	/** Sequence number for connection **/
	protected Long myConnIDNumber = null;
	protected int mySeqNum;
	
	/**
	 * Overloaded Constructor for the SIPPacket.
	 * 
	 * @param   s is the UserID object representing the source
	 * @param   sl is the Location object representing the source
	 * @param   ad is the String description of the Source
	 * @param   seq is the sequence number of the connection
	 * @param   meth is the integer representing the method id
	 * @param   pack is the integer representing the packet type
	 * 
	 * @return  void
	 * 
	 */
	protected SIPPacket(UserID s, Location sl, String ad, Long id, int seq, int meth, int pack)
	{
		super(meth,pack);
		source = s;
		srcLoc = sl;		
		appDesc = ad;
		myConnIDNumber = id;
		mySeqNum = seq;
	}
	
	/**
	 * Returns the UserID associated with the sender of the SIPPacket
	 * 
	 * @param   None
	 * @return  UserID object of the sender
	 */
	public UserID getSenderID(){
		return source;
	}

	/**
	 * Returns the Location associated with the sender of the SIPPacket
	 * 
	 * @param   None
	 * @return  Location object of the sender
	 */
	public Location getSenderLoc(){
		return srcLoc;
	}
	
	/**
	 * Returns the description of the sender of the SIPPacket
	 * 
	 * @param   None
	 * @return  a String description of the sender
	 */
	public String getDescription(){
		return appDesc;
	}
	
	/**
	 * Returns the reason a packet was rejected
	 * 
	 * @param   None
	 * @return  a String describing the reason for rejecting the invitation
	 */
	protected String getReason(){
		return reason;
	}
	
	/**
	 * Returns the senders' PropertiesCollection
	 * 
	 * @param   None
	 * @return  senders' PropertiesCollection object
	 */	
	public PropertiesCollection getPropertiesCollection(){
		return myPC;
	}

	/**
	 * Assigns the source UserID
	 * 
	 * @param   s is the UserID object representing the sender
	 * @return  void
	 */	
	protected void setSourceID(UserID s){
		source = s;
	}

	/**
	 * Assigns the source Location
	 * 
	 * @param   l is the Location object representing the sender's location
	 * @return  void
	 */		
	protected void setSourceLoc(Location l){
		srcLoc = l;
	}

	/**
	 * Sets the source description
	 * 
	 * @param   s is the String representing the sender's description
	 * @return  void
	 */		
	protected void setAppDescription(String s){
		appDesc = s;
	}
	
	/**
	 * Assigns the source's PropertiesCollection for Data Exchange
	 * 
	 * @param   pc is the PropertiesCollection object representing the sender's PropertiesCollection
	 * @return  void
	 */		
	protected void setPropertiesCollection(PropertiesCollection pc){
		myPC = pc;
	}

	/**
	 * Assigns the sequence number for the packet. A sequence number is exchanged in the 3-way handshake
	 * 
	 * @param   s is the Long object representing the sequence number
	 * @return  void
	 */		
	protected void setIDNumber(Long s){
		myConnIDNumber = s;
	}
	
	protected void setSeqNum(int s)
	{
		mySeqNum = s;
	}
	
	/**
	 * Returns the sequence number associated with the InvitePacket
	 * 
	 * @param   None
	 * @return  the Long object representing the sequence number
	 */
	public Long getIDNumber(){
		return myConnIDNumber;
	}
	
	public int getSeqNum()
	{
		return mySeqNum;
	}
	
	/**
	 * Applications call this method to indicate that they accept the invitation.
	 * 
	 * @param   None
	 * @return  void
	 * 
	 */	
	public void accept(){
		resultFlag = SignalID.ACCEPT;
	}

	/**
	 * Applications call this method to indicate that they reject the invitation.
	 * 
	 * @param   r is the reason for rejection
	 * @return  void
	 * 
	 */		
	public void reject(String r){
		resultFlag = SignalID.REJECT;
		reason = r;
	}
	
	/**
	 * Applications call this method to indicate that they are busy at present and cannot accept 
	 * the invitation.
	 * 
	 * @param   None
	 * @return  void
	 * 
	 */		
	public void busy(){
		resultFlag = SignalID.BUSY;
	}
	
	/**
	 * Applications call this method to confirm a sequence.
	 * 
	 * @param   None
	 * @return  void
	 * 
	 */
	public void confirm(){
		resultFlag = SignalID.CONFIRMED;
	}
	
	/**
	 * Checks whether the Invitation sent was accepted by the peer application.
	 * 
	 * @param   None.
	 * @return	TRUE if invitation was accepted, FALSE otherwise 
	 * 
	 */
	protected boolean wasAccepted(){
		return (resultFlag == SignalID.ACCEPT);
	}


	/**
	 * Checks whether the Invitation was rejected by the peer application.
	 * 
	 * @param   None.
	 * @return	TRUE if invitation was rejected, FALSE otherwise 
	 * 
	 */
	protected boolean wasRejected(){
		return (resultFlag == SignalID.REJECT);
	}	

	/**
	 * Checks whether the peer application was busy.
	 * 
	 * @param   None.
	 * @return	TRUE if peer was busy, FALSE otherwise 
	 * 
	 */
	protected boolean wasBusy(){
		return (resultFlag == SignalID.BUSY);
	}
	
	/**
	 * Checks whether it is a confirm packet .
	 * 
	 * @param   None.
	 * @return	TRUE if it is a confirm packet, FALSE otherwise 
	 * 
	 */
	protected boolean wasConfirmed(){
		return (resultFlag == SignalID.CONFIRMED);
	}
}